<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Brand;
use App\Models\Category;
use App\Models\Product;

class PageController extends Controller
{
    /**
     * Display the home page.
     *
     * @return \Illuminate\View\View
     */
    public function home()
    {
        $brands = Brand::all();
        $categories = Category::with('children')->whereNull('parent_id')->get();
        $products = Product::with('category')->where('is_active', 1)->take(10)->get();
        
        return view('pages.home', compact('brands', 'categories', 'products'));
    }

    /**
     * Display the products page.
     *
     * @return \Illuminate\View\View
     */
    public function products(Request $request)
    {
        $query = Product::with(['brand', 'category', 'primaryImage', 'reviews'])
            ->where('is_active', 1);

        // Filter by category
        if ($request->filled('category')) {
            $query->where('category_id', $request->category);
        }

        // Filter by brand
        if ($request->filled('brand')) {
            $query->where('brand_id', $request->brand);
        }

        // Search by name
        if ($request->filled('search')) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        // Get products with pagination
        $products = $query->paginate(12)->withQueryString();

        // Get categories and brands for filters
        $categories = Category::with('children')->whereNull('parent_id')->get();
        $brands = Brand::all();

        return view('pages.products', compact('products', 'categories', 'brands'));
    }

    /**
     * Display the about page.
     *
     * @return \Illuminate\View\View
     */
    public function about()
    {
        return view('pages.about');
    }

    /**
     * Display the contact page.
     *
     * @return \Illuminate\View\View
     */
    public function contact()
    {
        return view('pages.contact');
    }

    /**
     * Display the cart page.
     *
     * @return \Illuminate\View\View
     */
    public function cart()
    {
        return view('pages.cart');
    }

    /**
     * Display the checkout page.
     *
     * @return \Illuminate\View\View
     */
    public function checkout()
    {
        return view('pages.checkout');
    }

    /**
     * Display the wishlist page.
     *
     * @return \Illuminate\View\View
     */
    public function wishlist()
    {
        return view('pages.wishlist');
    }

    /**
     * Display the track order page.
     *
     * @return \Illuminate\View\View
     */
    public function track()
    {
        return view('pages.track');
    }

    /**
     * Display the brand detail page.
     *
     * @return \Illuminate\View\View
     */
    public function brand_detail()
    {
        return view('pages.brand_detail');
    }

    /**
     * Display the careers page.
     *
     * @return \Illuminate\View\View
     */
    public function careers()
    {
        return view('pages.careers');
    }

    /**
     * Display the offer page.
     *
     * @return \Illuminate\View\View
     */
    public function offer()
    {
        return view('pages.offer');
    }

    /**
     * Display the privacy policy page.
     *
     * @return \Illuminate\View\View
     */
    public function privacy_policy()
    {
        return view('pages.privacy-policy');
    }

    /**
     * Display the product list page.
     *
     * @return \Illuminate\View\View
     */
    public function product_list()
    {
        return view('pages.product-list');
    }

    /**
     * Display the product detail page.
     *
     * @return \Illuminate\View\View
     */
    public function product_detial(Request $request, $id = null)
    {
        if ($id) {
            // If ID is provided, show specific product detail
            $product = Product::with(['brand', 'category', 'images', 'reviews.user'])
                ->where('is_active', 1)
                ->findOrFail($id);
            
            // Get related products from the same category
            $relatedProducts = Product::with(['brand', 'category', 'primaryImage', 'reviews'])
                ->where('is_active', 1)
                ->where('category_id', $product->category_id)
                ->where('id', '!=', $product->id)
                ->take(3)
                ->get();
            
            return view('pages.product-detial', compact('product', 'relatedProducts'));
        }
        
        // Original logic for showing multiple products
        $query = Product::with(['brand', 'category', 'images', 'reviews.user'])
            ->where('is_active', 1);

        // Filter by category if provided
        if ($request->filled('category')) {
            $query->where('category_id', $request->category);
        }

        // Get 4 products for display
        $products = $query->take(4)->get();

        // Get categories and brands for filters
        $categories = Category::with('children')->whereNull('parent_id')->get();
        $brands = Brand::all();

        return view('pages.product-detial', compact('products', 'categories', 'brands'));
    }

    /**
     * Display the thank you page.
     *
     * @return \Illuminate\View\View
     */
    public function thank_you()
    {
        return view('pages.thank-you');
    }
}
