<?php
namespace App\Http\Controllers;

use App\Models\CartItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Response;

class CartController extends Controller
{
    // list user's cart items
    public function index()
    {
        $user = Auth::user();
        if (! $user) {
            return Response::json(['message' => 'Unauthenticated'], 401);
        }

        return CartItem::with('product')->where('user_id', $user->id)->get();
    }

    // add or update a cart item (create or update quantity)
    public function store(Request $r)
    {
        $user = Auth::user();
        if (! $user) {
            return Response::json(['message' => 'Unauthenticated'], 401);
        }

        $data = $r->validate([
            'product_id' => 'required|exists:products,id',
            'quantity'   => 'nullable|integer|min:1'
        ]);

        $cart = CartItem::updateOrCreate(
            ['user_id' => $user->id, 'product_id' => $data['product_id']],
            ['quantity' => $data['quantity'] ?? 1]
        );

        return Response::json($cart, 201);
    }

    // update quantity for a cart item (only owner can)
    public function update(Request $r, $id)
    {
        $user = Auth::user();
        if (! $user) {
            return Response::json(['message' => 'Unauthenticated'], 401);
        }

        $item = CartItem::findOrFail($id);

        // ownership check (replace authorize)
        if ($item->user_id !== $user->id) {
            return Response::json(['message' => 'Forbidden'], 403);
        }

        $item->update($r->validate(['quantity' => 'required|integer|min:1']));

        return Response::json($item);
    }

    // delete cart item (only owner can)
    public function destroy($id)
    {
        $user = Auth::user();
        if (! $user) {
            return Response::json(['message' => 'Unauthenticated'], 401);
        }

        $item = CartItem::findOrFail($id);

        // ownership check
        if ($item->user_id !== $user->id) {
            return Response::json(['message' => 'Forbidden'], 403);
        }

        $item->delete();

        return Response::json(null, 204);
    }
}
