import { DynamoDBClient } from "@aws-sdk/client-dynamodb";
import { GetCommand, PutCommand, UpdateCommand } from "@aws-sdk/lib-dynamodb";

const client = new DynamoDBClient({ region: "ap-south-1" });

const formatDate = (date) => {
    const day = String(date.getDate()).padStart(2, '0');
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const year = date.getFullYear();
    return `${day}-${month}-${year}`;
};

const addDays = (date, days) => {
    const result = new Date(date);
    result.setDate(result.getDate() + days);
    return result;
};

export const handler = async (event) => {
    const {
        v1: uid,
        v2: type,
        v3: location,
        v4: subid,
        v5: startDate,
        v6: email,
        v7: contact,
        v8: amount,
        v9: currency,
        v10: payment_id,
        v11: description,
        v12: days
    } = event;

    const dayCount = parseInt(days);
    const startDateObject = new Date(startDate.split('-').reverse().join('-'));
    const endDateObject = addDays(startDateObject, dayCount);
    const formattedStartDate = formatDate(startDateObject);
    const formattedEndDate = formatDate(endDateObject);
    const todayDateString = formatDate(new Date());

    const primaryKey = { subscribers: uid };

    try {
        // Step 0: Fetch username from 'eortv' table
        const userFromEortv = await client.send(new GetCommand({
            TableName: "eortv",
            Key: {
                eortv_media: uid
            }
        }));

        const username = userFromEortv?.Item?.username || "";

        // Step 1: Check if user already subscribed
        const existingItem = await client.send(new GetCommand({
            TableName: "SubscribedUsers",
            Key: primaryKey
        }));

        if (existingItem?.Item) {
            // Update subscription - Resubscriber
            const updateParams = {
                TableName: "SubscribedUsers",
                Key: primaryKey,
                UpdateExpression: `
                    SET #amt = :amount,
                        #cc = :countrycode,
                        #cur = :currency,
                        #email = :email,
                        #end = :enddate,
                        #loc = :location,
                        #name = :name,
                        #num = :number,
                        #pt = :paidthrough,
                        #pid = :paymentID,
                        #stype = :subscriptiontype,
                        #sid = :subscriptionID,
                        #resub = :resub,
                        #type = :type,
                        #start = :startdate
                `,
                ExpressionAttributeNames: {
                    "#amt": "amount",
                    "#cc": "countrycode",
                    "#cur": "currency",
                    "#email": "email",
                    "#end": "enddate",
                    "#loc": "location",
                    "#name": "name",
                    "#num": "number",
                    "#pt": "paidthrough",
                    "#pid": "paymentID",
                    "#stype": "subscriptiontype",
                    "#sid": "subscriptionID",
                    "#resub": "resub",
                    "#type": "type",
                    "#start": "startdate"
                },
                ExpressionAttributeValues: {
                    ":amount": amount.toString(),
                    ":countrycode": "+91",
                    ":currency": currency,
                    ":email": email,
                    ":enddate": formattedEndDate,
                    ":location": location,
                    ":name": username,
                    ":number": contact.toString(),
                    ":paidthrough": type,
                    ":paymentID": payment_id,
                    ":subscriptiontype": "auto",
                    ":subscriptionID": subid,
                    ":resub": "Resubscriber",
                    ":type": `${dayCount} Days`,
                    ":startdate": formattedStartDate
                }
            };

            await client.send(new UpdateCommand(updateParams));
            console.log("Item updated successfully.");
        } else {
            // Insert new subscription - New
            const putParams = {
                TableName: "SubscribedUsers",
                Item: {
                    subscribers: uid,
                    amount: amount.toString(),
                    countrycode: "+91",
                    currency: currency,
                    email: email,
                    enddate: formattedEndDate,
                    location: location,
                    name: username,
                    number: contact.toString(),
                    paidthrough: type,
                    paymentID: payment_id,
                    subscriptiontype: "auto",
                    subscriptionID: subid,
                    resub: "New",
                    type: `${dayCount} Days`,
                    startdate: formattedStartDate,
                }
            };

            await client.send(new PutCommand(putParams));
            console.log("New item inserted.");
        }

        // Step 4: Update 'newsubscriptions' table
        const existingNewSubs = await client.send(new GetCommand({
            TableName: "newsubscriptions",
            Key: {
                details: formattedStartDate
            }
        }));

        if (existingNewSubs?.Item) {
            const updateNewSubscribersParams = {
                TableName: "newsubscriptions",
                Key: {
                    details: formattedStartDate
                },
                UpdateExpression: `SET #uid = :userData`,
                ExpressionAttributeNames: {
                    "#uid": uid
                },
                ExpressionAttributeValues: {
                    ":userData": {
                        amount: amount.toString(),
                        currency: currency,
                        emailid: email,
                        startdate: formattedStartDate,
                        enddate: formattedEndDate,
                        location: location,
                        payid: payment_id,
                        type: `${dayCount} Days`,
                        uid: uid,
                        notes: "",
                        paidthrough: type
                    }
                }
            };

            await client.send(new UpdateCommand(updateNewSubscribersParams));
            console.log("newsubscriptions table updated for existing date.");
        } else {
            const putNewSubscribersParams = {
                TableName: "newsubscriptions",
                Item: {
                    details: formattedStartDate,
                    [uid]: {
                        amount: amount.toString(),
                        currency: currency,
                        emailid: email,
                        startdate: formattedStartDate,
                        enddate: formattedEndDate,
                        location: location,
                        payid: payment_id,
                        type: `${dayCount} Days`,
                        uid: uid,
                        notes: "",
                        paidthrough: type
                    }
                }
            };

            await client.send(new PutCommand(putNewSubscribersParams));
            console.log("newsubscriptions table item created for today's date.");
        }

        // Step 5: Update or insert into 'subscriptioncheck' table
        const subCheckKey = { details: subid }; // Now partition key is subid
        const existingCheck = await client.send(new GetCommand({
            TableName: "subscriptioncheck",
            Key: subCheckKey
        }));

        if (existingCheck?.Item) {
            // Update existing
            const updateSubCheckParams = {
                TableName: "subscriptioncheck",
                Key: subCheckKey,
                UpdateExpression: `SET #d = :days, #uid = :uid`,
                ExpressionAttributeNames: {
                    "#d": "days",
                    "#uid": "uid"
                },
                ExpressionAttributeValues: {
                    ":days": days,
                    ":uid": uid
                }
            };
            await client.send(new UpdateCommand(updateSubCheckParams));
            console.log("subscriptioncheck table updated.");
        } else {
            // Create new
            const putSubCheckParams = {
                TableName: "subscriptioncheck",
                Item: {
                    details: subid, // Partition key now uses subid
                    days: days,
                    uid: uid         // Store uid under the field 'uid'
                }
            };
            await client.send(new PutCommand(putSubCheckParams));
            console.log("subscriptioncheck table entry created.");
        }


        return {
            statusCode: 200,
            body: JSON.stringify({ message: "Success" })
        };

    } catch (error) {
        console.error("Error processing record: ", error);
        return {
            statusCode: 500,
            body: JSON.stringify({
                message: "Error processing record",
                error: error.message
            }),
        };
    }
};
