import { DynamoDBClient } from "@aws-sdk/client-dynamodb";
import { GetCommand, PutCommand, UpdateCommand } from "@aws-sdk/lib-dynamodb";

// Initialize DynamoDB Client
const client = new DynamoDBClient({ region: "ap-south-1" });

// Helper function to format a Date object into 'DD-MM-YYYY'
const formatDate = (date) => {
    const day = String(date.getDate()).padStart(2, '0');
    const month = String(date.getMonth() + 1).padStart(2, '0');  // Months are zero-indexed in JS
    const year = date.getFullYear();
    return `${day}-${month}-${year}`;
};

// Helper function to add days to a date
const addDays = (date, days) => {
    const newDate = new Date(date);
    newDate.setDate(newDate.getDate() + days);
    return newDate;
};

export const handler = async (event) => {
    const { v1: uid, v2: type, v3: location, v4: rent, v5: rentid, v6: startDate, v7: email, v8: contact, v9: amount, v10: currency, v11: payment_id } = event;

    // Step 1: Fetch 'username' from the 'eortv' table using 'uid' as the partition key
    let username;
    try {
        const getParamsEortv = {
            TableName: "eortv",
            Key: {
                eortv_media: uid + "" // Assuming 'eortv_media' is the partition key, and uid is a string
            }
        };

        const getResultEortv = await client.send(new GetCommand(getParamsEortv));

        if (getResultEortv.Item && getResultEortv.Item.username) {
            username = getResultEortv.Item.username;  // Extract 'username' from the result
        } else {
            throw new Error(`Username not found for uid: ${uid}`);
        }
    } catch (error) {
        console.error("Error fetching username from eortv: ", error);
        return {
            statusCode: 500,
            body: JSON.stringify({
                message: "Error fetching username from eortv",
                error: error.message
            }),
        };
    }

    // Step 2: Fetch 'name' from the 'ALLSHOWS' table using 'rentid' as the partition key
    let showName;
    try {
        const getParamsAllshows = {
            TableName: "ALLSHOWS",
            Key: {
                details: rentid + "" // Assuming 'details' is the partition key, and rentid is a string
            }
        };

        const getResultAllshows = await client.send(new GetCommand(getParamsAllshows));

        if (getResultAllshows.Item && getResultAllshows.Item.name) {
            showName = getResultAllshows.Item.name;  // Extract 'name' from the result
        } else {
            throw new Error(`Show name not found for rentid: ${rentid}`);
        }
    } catch (error) {
        console.error("Error fetching show name from ALLSHOWS: ", error);
        return {
            statusCode: 500,
            body: JSON.stringify({
                message: "Error fetching show name from ALLSHOWS",
                error: error.message
            }),
        };
    }

    // Step 3: Format current date and time for 'details' (YYYY-MM-DD HH-MM-SS-SSS)
    const currentTimestamp = new Date().toISOString().replace(/T/, ' ').replace(/:/g, '-').split('.')[0];

    // Step 4: Parse startDate (v6) and calculate endDate (startDate + 30 days)
    const startDateObject = new Date(startDate.split('-').reverse().join('-'));  // Convert DD-MM-YYYY to YYYY-MM-DD
    const endDateObject = addDays(startDateObject, 30);

    // Step 5: Format the start and end dates as 'DD-MM-YYYY'
    const formattedStartDate = formatDate(startDateObject);
    const formattedEndDate = formatDate(endDateObject);

    // Step 6: Construct the parameters for the PutCommand to insert the record into the 'rentals' table
    const putParams = {
        TableName: "rentals",  // Your DynamoDB table name
        Item: {
            "details": currentTimestamp,  // String value
            "amount": amount.toString(),  // Make sure numeric fields are passed as strings if needed
            "currency": currency,
            "email": email,
            "enddate": formattedEndDate,
            "location": location,
            "number": contact,
            "paidthrough": type,
            "paymentID": payment_id,
            "startdate": formattedStartDate,  // Use v6 as start date
            "type": `Rental - ${showName}`,  // Use the extracted 'name' from ALLSHOWS as part of 'type'
            "name": username  // Add the 'username' as 'name' in the rentals table
        }
    };

    try {
        // Step 7: Execute the PutCommand to insert the record into the 'rentals' table
        const result = await client.send(new PutCommand(putParams));
        console.log("Success: ", result);
    } catch (error) {
        console.error("Error inserting record: ", error);
        return {
            statusCode: 500,
            body: JSON.stringify({
                message: "Error inserting record",
                error: error.message
            }),
        };
    }

    // Step 8: Construct the parameters for the UpdateCommand to update the 'eortv' table with the new 'rent + rentid' key
    const updateParams = {
        TableName: "eortv",  // Your DynamoDB table name
        Key: {
            eortv_media: uid + ""  // Partition key
        },
        UpdateExpression: `SET #rentKey = :endDateValue`,
        ExpressionAttributeNames: {
            "#rentKey": `rent${rentid}`  // Dynamically create the key, e.g., 'rent36'
        },
        ExpressionAttributeValues: {
            ":endDateValue": formattedEndDate  // Set the value as the calculated end date (e.g., '30-04-2025')
        }
    };

    try {
        // Step 9: Execute the UpdateCommand to update the 'eortv' table
        const updateResult = await client.send(new UpdateCommand(updateParams));
        console.log("Update Success: ", updateResult);

        return {
            statusCode: 200,
            body: JSON.stringify({
                message: "Record inserted and updated successfully"
            }),
        };
    } catch (error) {
        console.error("Error updating eortv table: ", error);
        return {
            statusCode: 500,
            body: JSON.stringify({
                message: "Error updating eortv table",
                error: error.message
            }),
        };
    }
};
