import { DynamoDBClient } from "@aws-sdk/client-dynamodb";
import { GetCommand, PutCommand, UpdateCommand } from "@aws-sdk/lib-dynamodb";

const client = new DynamoDBClient({ region: "ap-south-1" });

const formatDate = (date) => {
    const day = String(date.getDate()).padStart(2, '0');
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const year = date.getFullYear();
    return `${day}-${month}-${year}`;
};

const addDays = (date, days) => {
    const result = new Date(date);
    result.setDate(result.getDate() + days);
    return result;
};

export const handler = async (event) => {
    const {
        v1: subscription_id,
        v2: payment_id,
        v3: email,
        v4: contact,
        v5: description,
        v6: amount,
        v7: currency,
        v8: startDate
    } = event;

    // Step 1: Lookup from 'subscriptioncheck' table using subscription_id
    const checkCommand = new GetCommand({
        TableName: "subscriptioncheck",
        Key: { details: subscription_id }
    });

    const checkResult = await client.send(checkCommand);

    if (!checkResult?.Item) {
        console.log("No matching subscription found in subscriptioncheck.");
        return { statusCode: 404, body: "Subscription not found" };
    }

    const uid = checkResult.Item.uid;
    const days = parseInt(checkResult.Item.days);
    const startDateObject = new Date(startDate.split('-').reverse().join('-'));
    const endDateObject = addDays(startDateObject, days);
    const formattedStartDate = formatDate(startDateObject);
    const formattedEndDate = formatDate(endDateObject);

    // Step 1.5: Get existing SubscribedUsers record to extract extra fields
    let existingSubData;
    try {
        const existingSubCommand = new GetCommand({
            TableName: "SubscribedUsers",
            Key: { subscribers: uid + "" }
        });
        const existingSubResult = await client.send(existingSubCommand);
        existingSubData = existingSubResult?.Item;

        if (!existingSubData) {
            console.log("No existing SubscribedUsers data found for uid:", uid);
            return { statusCode: 404, body: "Existing user data not found" };
        }
    } catch (err) {
        console.error("Failed fetching existing SubscribedUsers:", err);
        return { statusCode: 500, body: "Error fetching current subscription data" };
    }

    const location = existingSubData.location || "Not Found";
    const paidthrough = existingSubData.paidthrough || "Not Found";

    // Step 2: Update SubscribedUsers table using uid
    const updateParams = {
        TableName: "SubscribedUsers",
        Key: { subscribers: uid + "" },
        UpdateExpression: `SET 
            amount = :amount,
            countrycode = :countrycode,
            currency = :currency,
            email = :email,
            enddate = :enddate,
            #number = :number,
            paymentID = :paymentID,
            subscriptiontype = :subscriptiontype,
            subscriptionID = :subscriptionID,
            resub = :resub,
            #type = :type,
            startdate = :startdate`,
        ExpressionAttributeNames: {
            "#number": "number",
            "#type": "type"
        },
        ExpressionAttributeValues: {
            ":amount": amount.toString(),
            ":countrycode": "+91",
            ":currency": currency,
            ":email": email,
            ":enddate": formattedEndDate,
            ":number": contact.toString(),
            ":paymentID": payment_id,
            ":subscriptiontype": "auto",
            ":subscriptionID": subscription_id,
            ":resub": "Resubscriber",
            ":type": `${days} Days`,
            ":startdate": startDate
        }
    };

    try {
        await client.send(new UpdateCommand(updateParams));
        console.log("SubscribedUsers table updated successfully.");

        // Step 3: Upsert into newsubscriptions table
        const newSubscriptionsParams = {
            TableName: "newsubscriptions",
            Key: { details: formattedStartDate },
            UpdateExpression: `SET #uid = :userData`,
            ExpressionAttributeNames: {
                "#uid": uid
            },
            ExpressionAttributeValues: {
                ":userData": {
                    amount: amount.toString(),
                    currency: currency,
                    emailid: email,
                    startdate: formattedStartDate,
                    enddate: formattedEndDate,
                    location: location,
                    payid: payment_id,
                    type: `${days} Days`,
                    uid: uid,
                    notes: description,
                    paidthrough: paidthrough
                }
            }
        };

        await client.send(new UpdateCommand(newSubscriptionsParams));
        console.log("newsubscriptions table upserted successfully.");

        return { statusCode: 200, body: "SubscribedUsers and newsubscriptions updated" };

    } catch (error) {
        console.error("Update failed:", error.name, error.message, error.stack);
        return { statusCode: 500, body: `Failed to update tables: ${error.message}` };
    }
};
